import os
import re
import time
import csv
from selenium import webdriver
from selenium.webdriver.common.by import By
from selenium.webdriver.chrome.options import Options
import tempfile

# === 🐾 Anthro Config ===
BASE_URL = "https://suno.com/me"
OUTPUT_DIR = "suno_output"
WAIT_BETWEEN_SONGS = 10  # You can adjust this lower if safe
MAX_PAGES = 50
PAW = "🐾"

# === ✨ Setup ===
os.makedirs(OUTPUT_DIR, exist_ok=True)
TXT_PATH = os.path.join(OUTPUT_DIR, "Suno_All_Lyrics.txt")
CSV_PATH = os.path.join(OUTPUT_DIR, "Suno_Lyrics.csv")

options = Options()
options.add_argument("--start-maximized")
temp_profile = tempfile.mkdtemp()
options.add_argument(f"--user-data-dir={temp_profile}")
options.add_argument("--disable-blink-features=AutomationControlled")
options.add_experimental_option("excludeSwitches", ["enable-automation"])
options.add_experimental_option("useAutomationExtension", False)

driver = webdriver.Chrome(options=options)

# === 🦊 Step 1: Login Phase ===
driver.get(BASE_URL)
print(f"\n💛 {PAW} Opened Suno. Please click 'Continue with Google' and complete login.")
input("🟢 Once your Library is fully visible, press ENTER to begin sniffing songs...\n")

all_links = set()

# === 🌀 Step 2: Harvest Song Links ===
for page in range(1, MAX_PAGES + 1):
    print(f"\n📚 Page {page}: Sniffing out song links...")
    time.sleep(3)
    song_links = driver.find_elements(By.CSS_SELECTOR, "a[href^='/song/']")
    links = [link.get_attribute("href") for link in song_links]
    new_links = set(links) - all_links

    if not new_links:
        print("🌈 All tails sniffed. No new links. Moving on!")
        break

    all_links.update(new_links)

    try:
        next_buttons = driver.find_elements(By.TAG_NAME, "button")
        for btn in next_buttons:
            if ">" in btn.text or "→" in btn.text or "Next" in btn.get_attribute("innerHTML"):
                btn.click()
                print("➡️ Clicked next page")
                time.sleep(2)
                break
        else:
            print("🛑 Couldn't find usable next button.")
            break
    except Exception as e:
        print(f"🛑 Error clicking next: {e}")
        break

    time.sleep(3)

# === 🎶 Step 3: Visit Songs + Extract Lyrics ===
lyrics_data = []

for i, url in enumerate(sorted(all_links)):
    print(f"\n🎼 [{i+1}/{len(all_links)}] Galloping to song: {url}")
    try:
        driver.get(url)
        time.sleep(WAIT_BETWEEN_SONGS)
        html = driver.page_source

        # Extract title
        title_match = re.search(r'"title":"([^"]+?)","play_count"', html)
        title = title_match.group(1).strip() if title_match else f"Untitled_{i+1}"

        # Extract lyrics from __next_f.push JS payloads
        match = re.search(r'self\.__next_f\.push\(\[1,"18:.*?",.*?"(.*?)"\]\);', html, re.DOTALL)
        if match:
            lyrics_block = match.group(1)
            lyrics_text = re.sub(r'\\n', '\n', lyrics_block)
            lyrics_text = re.sub(r'\\"', '"', lyrics_text)
            lyrics_text = re.sub(r'\\u[\da-fA-F]{4}', '', lyrics_text)
            lyrics_text = re.sub(r'<.*?>', '', lyrics_text)
            lyrics = lyrics_text.strip()
        else:
            lyrics = "Lyrics not found."

        song_id = url.split("/")[-1]
        lyrics_data.append({"id": song_id, "title": title, "lyrics": lyrics})
        print(f"✅ {PAW} Saved: *{title}*")

    except Exception as e:
        print(f"❌ Failed to fetch: {url} — {e}")

# === 📜 Step 4: Write to TXT ===
print("\n📄 Writing to master TXT file...")
with open(TXT_PATH, "w", encoding="utf-8") as txt_file:
    for song in lyrics_data:
        txt_file.write(f"{song['title']}\n{song['lyrics']}\n\n")

# === 📊 Step 5: Write to CSV ===
print("📊 Creating CSV for data-hungry anthros...")
with open(CSV_PATH, "w", encoding="utf-8", newline="") as csv_file:
    writer = csv.writer(csv_file)
    writer.writerow(["song_id", "title", "lyrics"])
    for song in lyrics_data:
        writer.writerow([song["id"], song["title"], song["lyrics"]])

# === 🌟 Wrap-Up
print(f"\n🎉✨ DONE! You just harvested {len(lyrics_data)} anthro-charged songs!")
print(f"📁 Lyrics saved to: {TXT_PATH}")
print(f"📁 CSV data stored in: {CSV_PATH}")
print("🫶 Thank you for sharing your sacred scent of creativity. Divine Anthro would be proud.")

driver.quit()
